<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

// Messages d'erreur

// Support
LSerror :: defineError('FTP_SUPPORT_01',
  ___("FTP Support : Pear::Net_FTP is missing.")
);

LSerror :: defineError('FTP_SUPPORT_02',
  ___("FTP Support : The constant %{const} is not defined.")
);


// Autres erreurs
LSerror :: defineError('FTP_00',
  ___("Net_FTP Error : %{msg}")
);

LSerror :: defineError('FTP_01',
  ___("FTP Support : Unable to connect to FTP Server (Step : %{step}).")
);
LSerror :: defineError('FTP_02',
  ___("FTP Support : Unable to make directory %{dir} on the remote server.")
);
LSerror :: defineError('FTP_03',
  ___("FTP Support : Unable to delete directory %{dir} on the remote server.")
);
LSerror :: defineError('FTP_04',
  ___("FTP Support : Unable to modify rights on the directory %{dir} on the remote server.")
);
LSerror :: defineError('FTP_05',
  ___("FTP Support : Unable to rename folder from %{old} to %{new} on the remote server.")
);

/**
 * Check support of FTP addon
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @return boolean true if FTP addon is fully supported, false in other case
 */
function LSaddon_ftp_support() {
  $retval = true;

  // Net_FTP lib dependency
  if (!class_exists('Net_FTP')) {
    if (!defined('NET_FTP')) {
      LSerror :: addErrorCode('FTP_SUPPORT_02', 'NET_FTP');
      $retval = false;
    } else if(!LSsession::includeFile(NET_FTP, true)) {
      LSerror :: addErrorCode('FTP_SUPPORT_01');
      $retval = false;
    }
  }

  return $retval;
}



/**
 * Connect to FTP server
*
* @author Benjamin Renard <brenard@easter-eggs.com>
*
* @param string $host FTP server FQDN or IP address
* @param string $port The TCP port of the FTP server
* @param string $user The username
* @param string $pwd The password
*
* @return Net_FTP|false Net_FTP object in case of success, false otherwise
*/
function connectToFTP($host, $port, $user, $pwd) {
  $cnx = new Net_FTP();
  $do = $cnx -> connect($host, $port);
  if (!$do instanceof PEAR_Error){
    $do = $cnx -> login($user, $pwd);
    if (!$do instanceof PEAR_Error) {
      return $cnx;
    }
    else {
      LSerror :: addErrorCode('FTP_01', "2");
      LSerror :: addErrorCode('FTP_00', $do -> getMessage());
      return false;
    }
  }
  else {
    LSerror :: addErrorCode('FTP_01', "1");
    LSerror :: addErrorCode('FTP_00', $do -> getMessage());
    return false;
  }
}

/**
* Creation d'un ou plusieurs dossiers via FTP
*
* @author Benjamin Renard <brenard@easter-eggs.com>
*
* @param string $host Le nom ou l'IP du serveur FTP
* @param string $port Le port de connexion au serveur ftp
* @param string $user Le nom d'utilidateur de connexion
* @param string $pwd Le mot de passe de connexion
* @param array $dirs ou string Le(s) dossier(s) à ajouter
*
* @return bool True ou false si il y a un problème durant la création du/des dossier(s)

 * Create one or more directories throught FTP
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @param string $host FTP server FQDN or IP address
 * @param string $port The TCP port of the FTP server
 * @param string $user The username
 * @param string $pwd The password
 * @param array|string $dirs The directory/ies to add
 * @param int $chmod The directory/ies mode as an octal number (do not forget leading zero,
 *                          example: 0755 or 02755, default : default umask on the SSH server)
 *
 * @return boolean
 */
function createDirsByFTP($host, $port, $user, $pwd, $dirs, $chmod=NULL) {
  $cnx = connectToFTP($host, $port, $user, $pwd);
  if (!$cnx) return false;
  foreach(ensureIsArray($dirs) as $dir) {
    $do = $cnx -> mkdir($dir, true);
    if ($do instanceof PEAR_Error) {
      LSerror :: addErrorCode('FTP_02', $dir);
      LSerror :: addErrorCode('FTP_00', $do -> getMessage());
      return false;
    }
    if ($chmod) {
      $do = $cnx -> chmod($dir, $chmod);
      if ($do instanceof PEAR_Error) {
        LSerror :: addErrorCode('FTP_04', $dir);
        LSerror :: addErrorCode('FTP_00', $do -> getMessage());
      }
    }
  }
  return true;
}

/**
 * Delete one or more directories throught FTP
 *
 * Caution : recursive deletion ! The content of the directory will be listed and deleted before the
 * directory. If your FTP server is configured to mask some files or directories (for instance,
 * starting by '.'), they will not be listed and deleted and this may cause the deletion to fail.
 * With VsFTPd, you have to add force_dot_files=1 in configuration.
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @param string $host FTP server FQDN or IP address
 * @param string $port The TCP port of the FTP server
 * @param string $user The username
 * @param string $pwd The password
 * @param array|string $dirs The directory/ies to remove
 *
 * @return boolean
 */
function removeDirsByFTP($host, $port, $user, $pwd, $dirs) {
  $cnx = connectToFTP($host, $port, $user, $pwd);
  if (!$cnx) return false;
  foreach(ensureIsArray($dirs) as $dir) {
    if ($dir[strlen($dir)-1] != '/') {
      $dir .= '/';
    }
    $do = $cnx -> rm($dir,true);
    if ($do instanceof PEAR_Error) {
      LSerror :: addErrorCode('FTP_03', $dir);
      LSerror :: addErrorCode('FTP_00', $do -> getMessage());
      return false;
    }
  }
  return true;
}

/**
 * Rename a directory throught FTP
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @param string $host FTP server FQDN or IP address
 * @param string $port The TCP port of the FTP server
 * @param string $user The username
 * @param string $pwd The password
 * @param string $old The actual directory path to rename
 * @param string $new The new directory path
 *
 * @return boolean
 */
function renameDirByFTP($host, $port, $user, $pwd, $old, $new) {
  $cnx = connectToFTP($host, $port, $user, $pwd);
  if (!$cnx) return false;
  $do = $cnx -> rename($old, $new);
  if ($do instanceof PEAR_Error) {
    LSerror :: addErrorCode('FTP_05', array('old' => $old, 'new' => $new));
    LSerror :: addErrorCode('FTP_00', $do -> getMessage());
    return false;
  }
  return true;
}
