<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

// Error messages

// Support
LSerror :: defineError('MAIL_SUPPORT_01',
  ___("MAIL Support : Pear::MAIL is missing.")
);
LSerror :: defineError('MAIL_SUPPORT_02',
  ___("MAIL Support : Pear::MAIL_MIME is missing.")
);

// Other errors
LSerror :: defineError('MAIL_00',
  ___("MAIL Error : %{msg}")
);

LSerror :: defineError('MAIL_01',
  ___("MAIL : Error sending your email")
);

/**
 * Check support of this LSaddon
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @return boolean true if this LSaddon is fully supported, false otherwise
 */
function LSaddon_mail_support() {
  $retval=true;

  // Lib dependencies (check/load)
  if (!class_exists('Mail')) {
    if(!LSsession::includeFile(PEAR_MAIL, true)) {
      LSerror :: addErrorCode('MAIL_SUPPORT_01');
      $retval=false;
    }
  }

  if (!class_exists('Mail_mime')) {
    if(!LSsession::includeFile(PEAR_MAIL_MIME, true)) {
      LSerror :: addErrorCode('MAIL_SUPPORT_02');
      $retval=false;
    }
  }

  if ($retval)
    LScli :: add_command(
      'test_send_mail',
      'cli_test_send_mail',
      'Send a test email',
      "[-s subject] [-b body] [-H] [recipient1] [...]",
      array (
        "  -s/--subject     The test email subject (optional)",
        "  -b/--body        The test email body (optional)",
        "  -H/--HTML        Enable HTML email body mode (optional)",
        "  --header         Email header using format:",
        "                   header_name=header_value",
        "                   Multiple headers could be specified by using this",
        "                   optional argument multiple time.",
        "  -a|--attachment  Email attachment using format:",
        "                   /path/to/attachment.file[:filename]",
        "                   The filename is optional (default: using source filename).",
        "                   Multiple attachments could be specified by using this",
        "                   optional argument multiple time.",
        "  --bcc            Add Blind Carbon Copy (BCC) recipient(s)",
        "  --cc             Add Carbon Copy (CC) recipient(s)",
        "  recipients       The test email recipient(s) (required).",
      ),
      false,  // This command does not need LDAP connection
      'cli_test_send_mail_autocompleter'
    );

  return $retval;
}

/**
 * Send an email
 *
 * @param string|array<string> $to Email recipient(s)
 * @param string $subject Email subject
 * @param string $msg Email body
 * @param array<string,string>|null $headers Email headers
 * @param array<string,string>|null $attachments Email attachments as an array with
 *                             filepath as key and filename as value
 * @param string|null $eol End of line string (default : \n)
 * @param string|null $encoding Email encoding (default: utf8)
 * @param boolean $html Set to true to send an HTML email (default: false)
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 *
 * @return boolean true si MAIL est pleinement supporté, false sinon
 */
function sendMail($to, $subject, $msg, $headers=null, $attachments=null,
                  $eol=null, $encoding=null, $html=false) {
  global $MAIL_SEND_PARAMS, $MAIL_HEARDERS, $MAIL_CATCH_ALL;
  $mail_obj = Mail::factory(MAIL_SEND_METHOD, (isset($MAIL_SEND_PARAMS)?$MAIL_SEND_PARAMS:null));
  $logger = LSlog :: get_logger('LSaddon_mail');

  if (!$headers) $headers = array();
  if (isset($MAIL_HEARDERS) && is_array($MAIL_HEARDERS)) {
    $headers = array_merge($headers,$MAIL_HEARDERS);
  }
  $logger -> trace(
    'Mail catch all: '.(
      isset($MAIL_CATCH_ALL) && $MAIL_CATCH_ALL?
      varDump($MAIL_CATCH_ALL):'not set')
  );
  if (isset($MAIL_CATCH_ALL) && $MAIL_CATCH_ALL) {
    $logger -> debug(
      'Mail catch to '.
      (is_array($MAIL_CATCH_ALL)?implode(',', $MAIL_CATCH_ALL):$MAIL_CATCH_ALL)
    );
    $msg .= sprintf(
      (
        $html?
        _("</hr><p><small>Mail initialy intended for %s.</small></p>"):
        _("\n\n\nMail initialy intended for %s.")
      ),
      (is_array($to)?implode(',', $to):$to));
    $to = (
      is_array($MAIL_CATCH_ALL)?
      implode(',', $MAIL_CATCH_ALL):$MAIL_CATCH_ALL
    );
  }

  if (isset($headers['From'])) {
    $from = $headers['From'];
    unset($headers['From']);
  }
  else {
    $from = LSsession :: getEmailSender();
  }

  $headers["To"] = $to;

  $to = ensureIsArray($to);

  foreach(array_keys($headers) as $header) {
    if(in_array(strtoupper($header), array('BCC', 'CC'))) {
      if (isset($MAIL_CATCH_ALL) && $MAIL_CATCH_ALL) {
        $logger -> debug("Mail catched: remove $header header");
        $msg .= sprintf(
          (
            $html?
            _("<p><small>%s: %s</small></p>"):
            _("\n%s: %s")
          ),
          strtoupper($header),
          (is_array($headers[$header])?implode(',', $headers[$header]):$headers[$header]));
        unset($headers[$header]);
        continue;
      }
      $to = array_merge($to, $headers[$header]);
    }
  }

  if (!$encoding) $encoding = "utf8";
  $mime = new Mail_mime(
    array(
      'eol' => ($eol?$eol:"\n"),
      ($html?'html_charset':'text_charset') => $encoding,
      'head_charset' => $encoding,
    )
  );

  if ($from)
    $mime->setFrom($from);

  if ($subject)
    $mime->setSubject($subject);

  if ($html)
    $mime->setHTMLBody($msg);
  else
    $mime->setTXTBody($msg);

  if (is_array($attachments) && !empty($attachments)) {
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    foreach ($attachments as $file => $filename) {
      $mime->addAttachment($file, $finfo->file($file), $filename);
    }
  }

  $body = $mime->get();
  $headers = $mime->headers($headers);

  $ret = $mail_obj -> send($to, $headers, $body);

  if ($ret instanceof PEAR_Error) {
    LSerror :: addErrorCode('MAIL_01');
    LSerror :: addErrorCode('MAIL_00', $ret -> getMessage());
    return false;
  }
  return true;
}


if (php_sapi_name() != 'cli')
  return true;

/**
 * CLI command to send a test email
 * @param  array $command_args Command arguments
 * @return bool
 */
function cli_test_send_mail($command_args) {
  $recipients = array();
  $subject = "Test email";
  $body = "This is a test message.";
  $html = false;
  $headers = array();
  $attachments = array();
  for ($i=0; $i < count($command_args); $i++) {
    switch ($command_args[$i]) {
      case '--subject':
      case '-s':
        $i++;
        if (!isset($command_args[$i]))
          LScli :: usage("You must provide the email subject after -s/--subject parameter.");
        $subject = $command_args[$i];
        if (!$subject)
          LScli :: usage("Invalid subject provided.");
        break;

      case '--body':
      case '-b':
        $i++;
        if (!isset($command_args[$i]))
          LScli :: usage("You must provide the email body after -b/--body parameter.");
        $body = $command_args[$i];
        if (!$body)
          LScli :: usage("Invalid body provided.");
        break;

      case '--html':
      case '-H':
        $html = true;
        break;

      case '--header':
        $i++;
        LScli :: unquote_word($command_args[$i]);
        $parts = explode('=', $command_args[$i]);
        if (count($parts) != 2)
          LScli :: usage('Invalid header string ('.$command_args[$i].').');
        if (array_key_exists($parts[0], $headers))
          LScli :: usage('Header "'.$parts[0].'" already specified.');
        $headers[$parts[0]] = $parts[1];
        break;

      case '-a':
      case '--attachment':
        $i++;
        LScli :: unquote_word($command_args[$i]);
        $parts = explode(':', $command_args[$i]);
        $path = $parts[0];
        if (!is_file($path))
          LScli :: usage('Invalid attachment "'.$command_args[$i].'": file not found.');
        $attachments[$path] = count($parts) >= 2?$parts[1]:basename($path);
        break;

      case '--bcc':
        $i++;
        LScli :: unquote_word($command_args[$i]);
        if (!checkEmail($command_args[$i]))
          LScli :: usage('Invalid BCC recipient "'.$command_args[$i].'".');
        $headers['BCC'] = isset($headers['BCC'])?ensureIsArray($headers['BCC']):[];
        $headers['BCC'][] = $command_args[$i];
        break;

      case '--cc':
        $i++;
        LScli :: unquote_word($command_args[$i]);
        if (!checkEmail($command_args[$i]))
          LScli :: usage('Invalid CC recipient "'.$command_args[$i].'".');
          $headers['CC'] = isset($headers['CC'])?ensureIsArray($headers['CC']):[];
          $headers['CC'][] = $command_args[$i];
        break;

      default:
        if (checkEmail($command_args[$i]))
          $recipients[] = $command_args[$i];
        else
          LScli :: usage("Invalid parameter '".$command_args[$i]."'.");
    }
  }

  if (!$recipients)
    LScli :: usage("You must provide as least one email recipient as positional parameter");

  $logger = LSlog :: get_logger('LSaddon_mail');
  if (!sendMail($recipients, $subject, $body, $headers, $attachments, null, null, $html)) {
    $logger -> error("Fail to send test email sent to '".implode(', ', $recipients)."'.");
    return false;
  }
  $logger -> info("Test email sent to '".implode(', ', $recipients)."'.");
  return true;
}

/**
 * Args autocompleter for CLI test_send_mail command
 *
 * @param array<string> $comp_words List of already typed words of the command
 * @param int $comp_word_num The command word number to autocomplete
 * @param string $comp_word The command word to autocomplete
 * @param array<string> $opts List of global available options
 *
 * @return array<string> List of available options for the word to autocomplete
 **/
function cli_test_send_mail_autocompleter($comp_words, $comp_word_num, $comp_word, $opts) {
  if (isset($comp_words[$comp_word_num-1]))
    switch ($comp_words[$comp_word_num-1]) {
      case '-s':
      case '--subject':
      case '-b':
      case '--body':
      case '--header':
      case '-a':
      case '--attachment':
      case '--bcc':
      case '--cc':
        return array();
        break;
    }
  $opts = array_merge(
    $opts,
    array (
      '-s', '--subject',
      '-b', '--body',
      '-H', '--html',
      '--header',
      '-a', '--attachment',
      '--bcc', '--cc',
    )
  );
  return LScli :: autocomplete_opts($opts, $comp_word);
}
