<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ******************************************************************************/

/**
 * Règle de validation par comparaison de valeurs.
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 */
class LSformRule_compare extends LSformRule {

  /**
   * Validate values one by one or all together
   * @see LSformRule::validate_values()
   * @var boolean
   */
  const validate_one_by_one = False;

  // CLI parameters autocompleters
  protected static $cli_params_autocompleters = array(
    'operator' => null,
  );

  // Operators mapping
  static protected $_operators_aliases = array(
      '=='  => 'eq',
      '!='  => 'neq',
      '>'  => 'gt',
      '>='  => 'gte',
      '<'  => 'lt',
      '<='  => 'lte',
  );
  static protected $_operators_to_compare_function = array(
      'eq' => '_LSformRule_compare_eq',
      'neq' => '_LSformRule_compare_neq',
      'gt' => '_LSformRule_compare_gt',
      'gte' => '_LSformRule_compare_gte',
      'lt' => '_LSformRule_compare_lt',
      'lte' => '_LSformRule_compare_lte',
  );

  /**
   * Return the compare function associated with the specified operator
   *
   * @access private
   * @param  string $operator_name The operator name
   *
   * @return callable  The compare function
   */
  private static function _findOperatorCompareFunction($operator_name) {
    if (empty($operator_name))
      $operator_name = 'eq';
    elseif (isset(self :: $_operators_aliases[$operator_name]))
      $operator_name = self :: $_operators_aliases[$operator_name];
    elseif (!isset(self :: $_operators_to_compare_function[$operator_name]))
      $operator_name = 'eq';
    return self :: $_operators_to_compare_function[$operator_name];
  }

  /**
   * Validate form element values
   *
   * @param mixed $values The values to validate
   * @param array $options Validation options:
   *                       - Operator : $options['params']['operator'] allowed values
   * @param LSformElement &$formElement The related LSformElement object
   *
   * @return boolean True if value is valid, False otherwise
   */
  public static function validate($values, $options, &$formElement) {
    $operator = LSconfig :: get('params.operator', null, 'string', $options);
    if (!$operator) {
      LSerror :: addErrorCode('LSformRule_01',array('type' => 'compare', 'param' => 'operator'));
      return false;
    }
    $compareFn = self :: _findOperatorCompareFunction($operator);
    return $compareFn($values[0], $values[1]);
  }

}

function _LSformRule_compare_eq($a, $b) {return floatval($a) == floatval($b);}
function _LSformRule_compare_neq($a, $b) {return floatval($a) != floatval($b);}
function _LSformRule_compare_gt($a, $b) {return $a > $b;}
function _LSformRule_compare_gte($a, $b) {return $a >= $b;}
function _LSformRule_compare_lt($a, $b) {return $a < $b;}
function _LSformRule_compare_lte($a, $b) {return $a <= $b;}

# vim: tabstop=2 shiftwidth=2 softtabstop=2 expandtab
