<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

/**
 * LDAP attribute type compositeValueToJSON
 * Convert the composite attributes in the following format:
 *   [key1=value1][key2=value2]..
 * To JSON format (usable with LSattr_jsonCompositeAttribute):
 *   {"key1":"value1","key2":"value2"}
 */
class LSattr_ldap_compositeValueToJSON extends LSattr_ldap {

  /**
   * Return the display value of the attribute after handling it acording to its LDAP type
   *
   * @param mixed $data The attribute value
   *
   * @return array The display value of the attribute
   */
  public function getDisplayValue($data) {
    $ret = array();
    foreach(ensureIsArray($data) as $key => $val)
      $ret[$key] = json_encode(static :: parseValue($val));
    return $ret;
  }

  /**
   * Return the update value of the attribute after handling it acording to its LDAP type
   *
   * @param mixed $data The attribute value
   *
   * @return array The processed attribute value
   */
  public function getUpdateData($data) {
    $ret = array();
    foreach(ensureIsArray($data) as $key => $val)
      $ret[$key] = static :: encodeValue(json_decode($val, true));
    return $ret;
  }

  /**
   * Parse a value
   * @param  string $value  The value to parse
   * @return array|null The parse value as array, or null in case of error
   */
  public static function parseValue($value) {
    if (preg_match_all('/\[([^=]*)=([^\]]*)\]/', $value, $matches)) {
      $parseValue = array();
      for($i=0; $i<count($matches[0]); $i++) {
        $parseValue[$matches[1][$i]] = $matches[2][$i];
      }
      return $parseValue;
    }
    return null;
  }

  /**
   * Compute a composite value
   * @param  array $value The value to encode
   * @return string|false The computed composite value as string, or False in case of error
   */
  public static function encodeValue($value) {
    if (is_array($value)) {
      $ret = "";
      foreach($value as $key => $val)
        $ret .= "[$key=$val]";
      return $ret;
    }
    return False;
  }

}

# vim: tabstop=2 shiftwidth=2 softtabstop=2 expandtab
