<?php
/*******************************************************************************
 * Copyright (C) 2007 Easter-eggs
 * https://ldapsaisie.org
 *
 * Author: See AUTHORS file in top-level directory.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

******************************************************************************/

/**
 * Type d'attribut HTML select_object
 *
 * @author Benjamin Renard <brenard@easter-eggs.com>
 */
class LSattr_html_select_object extends LSattr_html{

  var $LSformElement_type = 'select_object';

  /**
   * Array of unrecognized values
   * @see LSattr_html_select_object::getFormValues()
   * @var array<int,string>
   */
  var $unrecognizedValues = array();

  /**
   * Ajoute l'attribut au formualaire passer en paramètre
   *
   * @param LSform &$form  Le formulaire
   * @param string $idForm L'identifiant du formulaire
   * @param array|string|null $data Valeur du champs du formulaire
   *
   * @return LSformElement|false L'element du formulaire ajouté, ou false
   */
  public function addToForm (&$form, $idForm, $data=NULL) {
    $this -> config['attrObject'] = $this;
    $element = $form -> addElement(
      $this -> LSformElement_type,
      $this -> name,
      $this -> getLabel(),
      $this -> config,
      $this
    );
    if(!$element) {
      LSerror :: addErrorCode('LSform_06', $this -> name);
      return false;
    }
    if ($data) {
      $values = $this -> getFormValues(ensureIsArray($data));
      if ($values) {
        $element -> setValue($values);
      }
    }
    return $element;
  }

  /**
   * Effectue les tâches nécéssaires au moment du rafraichissement du formulaire
   *
   * Récupère un array du type array('DNs' => 'displayName') à partir d'une
   * liste de DNs.
   *
   * @param mixed $data The attribute value (liste de DNs)
   *
   * @return array La valeur formatée de l'attribut (array('DNs' => 'displayName'))
   **/
  public function refreshForm($data, $fromDNs=false) {
    return $this -> getFormValues($data, $fromDNs);
  }

  /**
   * Check and return selectable objects configuration and eventually instanciate
   * it if &instanciate
   *
   * @param boolean &$instanciated_objects reference If this reference point to an array, each valid
   *                                                     selectable object type will be instanciated in this
   *                                                     array with object type name as key.
   *
   * @author Benjamin Renard <brenard@easter-eggs.com>
   *
   * @return array|false Selectable objects configuration or False on error.
   */
  public function getSelectableObjectsConfig(&$instanciated_objects) {
    $confs = $this -> getConfig('html_options.selectable_object');
    if (!is_array($confs)) {
      self :: log_debug('getSelectableObjectsConfig(): html_options.selectable_object not properly configured');
      return false;
    }

    // Make sure we have an array or configured selectable objects
    if (isset($confs['object_type'])) {
      $confs = array($confs);
    }

    // Return confs
    $ret_confs = array();

    // For each configured object types:
    // - check we have the minimal configuration (object_type & value_attribute)
    // - check we to not have multiple conf for the same object type
    // - load object type
    // - implement an object of this type (if is_array($instanciated_objects))
    // - if value are from attributes, check this attribute exists
    foreach ($confs as $conf) {
      if (!isset($conf['object_type'])) {
        LSerror :: addErrorCode(
          'LSattr_html_select_object_01',
          array(
            'attr' => $this -> name,
            'parameter' => 'object_type',
          )
        );
        return false;
      }

      if (!isset($conf['value_attribute'])) {
        LSerror :: addErrorCode(
          'LSattr_html_select_object_01',
          array(
            'attr' => $this -> name,
            'parameter' => 'value_attribute',
          )
        );
        return false;
      }

      if (array_key_exists($conf['object_type'], $ret_confs)) {
        LSerror :: addErrorCode('LSattr_html_select_object_04', array('type' => $conf['object_type'], 'attr' => $this -> name));
        return false;
      }
      $object_type = $conf['object_type'];

      if (!LSsession :: loadLSobject($object_type))
        return false;

      if (is_array($instanciated_objects))
        $instanciated_objects[$object_type] = new $object_type();

      if (!($conf['value_attribute']=='dn') && !($conf['value_attribute']=='%{dn}')) {
        if (!$object_type :: hasAttr($conf['value_attribute'])) {
          LSerror :: addErrorCode(
            'LSattr_html_select_object_02',
            array(
              'attr' => $this -> name,
              'object_type' => $conf['object_type'],
              'value_attribute' => $conf['value_attribute'],
            )
          );
          return false;
        }
      }

      // Handle other parameters
      $conf['display_name_format'] = LSconfig :: get('display_name_format', null, null, $conf);
      $conf['onlyAccessible'] = LSconfig :: get('onlyAccessible', false, 'bool', $conf);
      $conf['filter'] = LSconfig :: get('filter', null, null, $conf);

      $ret_confs[$object_type] = $conf;
    }
    return $ret_confs;
  }


  /**
   * Return an array of selected objects with DN as key and as value, an array with object name (key name)
   * and type (key object_type).
   *
   * @param mixed $values array|null Array of the input values ()
   * @param boolean $fromDNs boolean If true, considered provided values as DNs (default: false)
   * @param boolean $retrieveAttrValues boolean If true, final attribute values will be returned
   *                                            instead of selected objects info (default: false)
   *
   * @author Benjamin Renard <brenard@easter-eggs.com>
   *
   * @return array|false Array of selected objects with DN as key and object info as value or array
   *                     of attribute values if $retrieveAttrValues==true or False on error.
   */
  public function getFormValues($values=NULL, $fromDNs=false, $retrieveAttrValues=false) {
    self :: log_debug("getFormValues(): input values=".varDump($values));
    // Check parameters consistency
    if ($retrieveAttrValues && !$fromDNs) {
      self :: log_fatal('getFormValues(): $fromDNs must be true if $retrieveAttrValues==true');
      return false;
    }
    if (!is_array($values)) {
      self :: log_warning('getFormValues(): $values is not array');
      return false;
    }
    if (!LSsession :: loadLSclass("LSsearch"))
      return false;

    // Retrieve/check selectable objects config
    $objs = [];
    $confs = $this -> getSelectableObjectsConfig($objs);
    if (!is_array($confs) || empty($confs)) {
      self :: log_warning('getFormValues(): invalid selectable objects config');
      return false;
    }

    $selected_objects = [];
    $found_values = [];
    $unrecognizedValues = [];
    foreach ($confs as $conf) {
      $common_search_params = [
        "filter" => $conf['filter'],
        "attributes" => (
          $retrieveAttrValues && !in_array($conf['value_attribute'], ["dn", "%{dn}"])?
          [$conf['value_attribute']]:
          [$objs[$conf['object_type']]->rdn_attr]
        ),
        "displayFormat" => (
          $conf['display_name_format']?
          $conf['display_name_format']:
          $objs[$conf['object_type']] -> getDisplayNameFormat()
        ),
      ];
      foreach($values as $value) {
        // Ignore empty value and value already marked as unrecognized
        if (empty($value) || in_array($value, $unrecognizedValues))
          continue;

        // Compute search params based on value attribute type (DN or attribute valued) and $fromDNs
        if($fromDNs || in_array($conf['value_attribute'], ['dn', '%{dn}'])) {
          if (!checkDn($value)) {
            self :: log_warning(
              "getFormValues(): value '$value' is not a valid DN, pass"
            );
            continue;
          }
          if ($conf['onlyAccessible'] && !LSsession :: canAccess($conf['object_type'], $value)) {
            self :: log_debug(
              "getFormValues(): object {$conf['object_type']} '$value' is not accessible, pass"
            );
            continue;
          }
          $search_params = array_merge(
            $common_search_params,
            ["basedn" => $value, "scope" => "base"]
          );
        }
        else {
          $filter = Net_LDAP2_Filter::create($conf['value_attribute'], 'equals', $value);
          $search_params = array_merge(
            $common_search_params,
            [
              "filter" => (
                $common_search_params["filter"]?
                LSldap::combineFilters('and', [$common_search_params["filter"], $filter]):
                $filter
              ),
            ]
          );
        }

        // Search object
        $LSsearch = new LSsearch(
          $conf['object_type'],
          'LSattr_html_select_object::getFormValues',
          $search_params,
          true
        );

        if(!$LSsearch -> run(false)) {
          self :: log_warning(
            "getFormValues(): error during search of object(s) {$conf['object_type']} ".
            "for value '$value', pass"
          );
          continue;
        }

        $entries = $LSsearch -> listEntries();
        if (!is_array($entries) || empty($entries)) {
          self :: log_debug(
            "getFormValues(): value '$value' not found as {$conf['object_type']}, pass"
          );
          continue;
        }

        if (count($entries) > 1) {
          self :: log_warning(
            "getFormValues(): ".count($entries)." objects {$conf['object_type']} found ".
            "for value '$value', pass: ".implode(" / ", array_keys($entries))
          );
          if (array_key_exists($value, $selected_objects))
            unset($selected_objects[$value]);
          $unrecognizedValues[] = $value;
          $found_values[$value] = (
            array_key_exists($value, $found_values)?
            array_merge($found_values[$value], array_keys($entries)):
            array_keys($entries)
          );
          break;
        }
        $entry = $entries[key($entries)];
        self :: log_debug(
          "getFormValues(): value '$value' found as {$conf['object_type']}: {$entry->dn}"
        );

        // Check if it's the first this value match with an object
        if (array_key_exists($value, $found_values)) {
          // DN match with multiple object type
          LSerror :: addErrorCode(
            'LSattr_html_select_object_03',
            ['val' => $value, 'attribute' => $this -> name]
          );
          unset($selected_objects[$value]);
          $unrecognizedValues[] = $value;
          $found_values[$value][] = $entry->dn;
          break;
        }
        $found_values[$value] = [$entry->dn];

        if ($retrieveAttrValues) {
          // Retrieve attribute value case: $selected_objects[dn] = attribute value
          if(($conf['value_attribute']=='dn') || ($conf['value_attribute']=='%{dn}')) {
            $selected_objects[$entry->dn] = $entry->dn;
          }
          else {
            $val = ensureIsArray($entry -> get($conf['value_attribute']));
            if (!empty($val)) {
              $selected_objects[$entry->dn] = $val[0];
            }
            else {
              LSerror :: addErrorCode(
                'LSattr_html_select_object_06',
                array(
                  'name' => $entry -> displayName,
                  'attr' => $this -> name
                )
              );
            }
          }
        }
        else {
          // General case: $selected_objects[dn] = array(name + object_type)
          $selected_objects[$entry->dn] = array(
            'name' => $entry -> displayName,
            'object_type' => $conf['object_type'],
          );
          self :: log_debug(
            "getFormValues(): object {$conf['object_type']} info for value '$value': ".
            varDump($selected_objects[$entry->dn])
          );
        }
      }
    }

    // Retrieve attribute values case: return forged array values (list of attribute values)
    if ($retrieveAttrValues)
      return array_values($selected_objects);

    // General case
    $this -> unrecognizedValues = array_diff($values, array_keys($found_values));
    self :: log_debug("getFormValues(): unrecognizedValues=".varDump($this -> unrecognizedValues));
    self :: log_debug("getFormValues(): final values=".varDump($selected_objects));
    return $selected_objects;
  }

  /**
   * Get LSselect ID for this attribute
   *
   * @return string The LSselect ID for this attribute
   */
  public function getLSselectId() {
    $id = "";
    if ($this -> attribute -> ldapObject -> isNew())
      $id .= $this -> attribute -> ldapObject -> getType();
    else
      $id .= $this -> attribute -> ldapObject -> getDn();
    $id .= "|".$this -> name;
    return $id;
  }


  /**
   * Return array of attribute values form array of form values
   *
   * @param mixed $values Array of form values
   *
   * @author Benjamin Renard <brenard@easter-eggs.com>
   *
   * @return array|false Array of attribute values or False on error.
   */
  public function getValuesFromFormValues($values=NULL) {
    if (is_array($values)) {
      return $this -> getFormValues(array_keys($values), true, true);
    }
    return false;
  }

  /**
   * Return an array of selected objects with DN as key and display name as value
   * from LSselect
   *
   * @author Benjamin Renard <brenard@easter-eggs.com>
   *
   * @return array|false Tableau associatif des objects selectionnés avec en clé
   *                     le DN et en valeur ce qui sera affiché. False en cas d'erreur.
   */
  public function getValuesFromLSselect() {
    $LSselect_id = $this -> getLSselectId();
    if (LSsession :: loadLSclass('LSselect', null, true) && LSselect :: exists($LSselect_id)) {
      $selected_objects = LSselect :: getSelectedObjects($LSselect_id);
      self :: log_debug("getValuesFromLSselect(): selected objects retrieved from LSselect '$LSselect_id' = ".varDump($selected_objects));
      if (is_array($selected_objects)) {
        return $this -> getFormValues(
          array_keys($selected_objects),
          true
        );
      }
    }
    return false;
  }

  /**
   * Return the values to be displayed in the LSform
   *
   * @param array $data The values of attribute
   *
   * @return array The values to be displayed in the LSform
   **/
  public function getFormVal($data) {
    return $data;
  }

}

/*
 * Error Codes
 */
LSerror :: defineError('LSattr_html_select_object_01',
___("LSattr_html_select_object : parameter '%{parameter}' is missing (attribute : %{attr}).")
);
LSerror :: defineError('LSattr_html_select_object_02',
___("LSattr_html_select_object : the value of the parameter value_attribute in the configuration of the attribute %{attr} is incorrect. Object %{object_type} have no attribute %{value_attribute}.")
);
LSerror :: defineError('LSattr_html_select_object_03',
___("LSattr_html_select_object : more than one object returned corresponding to value %{val} of attribute %{attr}.")
);
LSerror :: defineError('LSattr_html_select_object_04',
___("LSattr_html_select_object : selection of object type %{type} is configured multiple time for attribute %{attr}.")
);
LSerror :: defineError('LSattr_html_select_object_05',
___("LSattr_html_select_object : the value '%{value}' seem to be duplicated in values of the attribute %{attr}.")
);
LSerror :: defineError('LSattr_html_select_object_06',
___("LSattr_html_select_object : selected object %{name} has no attribute %{attr} value, you can't select it.")
);

# vim: tabstop=2 shiftwidth=2 softtabstop=2 expandtab
